<?php
/**
* 2007-2021 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author PrestaShop SA <contact@prestashop.com>
*  @copyright  2007-2021 PrestaShop SA
*  @license    http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\PrestaShop\Core\Domain\Order\Exception\OrderException;
use PrestaShop\PrestaShop\Core\Payment\PaymentOption;
use Symfony\Component\HttpKernel\Log\Logger;
use Psr\Log\LogLevel;

class WpGateway extends PaymentModule
{
    public $apiMethod = array(
        'checkout' => '/api/v1/session',
        'refund'   => '/api/v1/payment/refund'
    );
    private $configPreset = array(
        'WPG_TITLE' => 'WPG Checkout',
        'WPG_MERCHANT_KEY' => '12345678-xxxx-xxxx-xxxx-123456789abc',
        'WPG_MERCHANT_PASSWORD' => 'xxccbbdd117646a30a18d74422d40eff',
        'WPG_CHECKOUT' => 'https://checkout.rafinita.com/',
        //'WPG_REFUND' => 'https://checkout.rafinita.com/api/v1/payment/refund',
        // 'WPG_METHOD' => array('card', 'googlepay','paypal')
    );
    public $logger;

    public function __construct()
    {
        $this->name = 'wpgateway';
        $this->version = '1.0.1';
        $this->author = 'softsprint';
        $this->tab = 'payments_gateways';

        $this->controllers = array('process', 'callback');

        $this->need_instance = 1;
        $this->ps_versions_compliancy = [
            'min' => '1.7.7',
            'max' => '1.7.99',
        ];

        $this->logger = new Logger(LogLevel::DEBUG, _PS_ROOT_DIR_ . '/var/logs/wpg.log');
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('WPG Checkout');
        $this->description = $this->l('Web Payment Gateway. The module works by opened checkout page, and then sending the details to payment system for verification.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');
    }

    public function install()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        foreach ($this->configPreset as $key => $value) {
            Configuration::updateValue($key, $value);
        }
        Configuration::updateValue('WPG_METHOD', serialize(array('card')));

        $ow_status = Configuration::get('WPG_STATE_WAITING');
        if ($ow_status === false) {
            $orderState = new OrderState();
        } else {
            $orderState = new OrderState((int)$ow_status);
        }


        $orderState->name = array();

        foreach (Language::getLanguages() as $language) {
            if (Tools::strtolower($language['iso_code']) == 'uk') {
                $orderState->name[$language['id_lang']] = 'Очікування завершення оплати';
            } else {
                $orderState->name[$language['id_lang']] = 'Awaiting for payment';
            }
        }

        $orderState->send_email  = false;
        $orderState->color       = '#FBA90C';
        $orderState->hidden      = false;
        $orderState->module_name = 'wpgateway';
        $orderState->delivery    = false;
        $orderState->logable     = false;
        $orderState->invoice     = false;
        $orderState->unremovable = true;
        $orderState->save();

        Configuration::updateValue('WPG_STATE_WAITING', (int)$orderState->id);


        return parent::install() &&
            $this->registerHook('PaymentOptions') &&
            $this->registerHook('ActionOrderSlipAdd');
    }

    public function uninstall()
    {
        $validator = true;

        // $orderStateId = Configuration::get('WPG_STATE_WAITING');
        // if ($orderStateId) {
        //     $orderState     = new OrderState();
        //     $orderState->id = $orderStateId;
        //     $orderState->delete();
        //     //unlink(_PS_IMG_DIR_ .'os/'.(int)$orderState->id.'.gif');
        // }

        //$validator &= Configuration::deleteByName('WPG_STATE_WAITING');


        foreach (array_keys($this->configPreset) as $key) {
            if (!Configuration::deleteByName($key)) {
                $validator &= false;
            }
        }
        $validator &= Configuration::deleteByName('WPG_METHOD');

        return $this->unregisterHook('PaymentOptions') &&
            $this->unregisterHook('ActionOrderSlipAdd') &&
            $validator &&
            parent::uninstall();
    }

    public function getContent()
    {
        $output = '';
        $title = Tools::getValue('WPG_TITLE');
        $merchant_key = Tools::getValue('WPG_MERCHANT_KEY');
        $merchant_password = Tools::getValue('WPG_MERCHANT_PASSWORD');
        $checkout_url = Tools::getValue('WPG_CHECKOUT');
        $payment_method = Tools::getValue('WPG_METHOD');

        $validator = true;

        if (Tools::isSubmit('submit' . $this->name)) {
            if (empty($title) || !Validate::isGenericName($title)) {
                $output = $this->displayError($this->l('Invalid title value'));
                $validator &= false;
            }
            //creds
            if (empty($merchant_key) || !Validate::isAnything($merchant_key)) {
                $output = $this->displayError($this->l('Invalid title value'));
                $validator &= false;
            }
            if (empty($merchant_password) || !Validate::isAnything($merchant_password)) {
                $output = $this->displayError($this->l('Invalid title value'));
                $validator &= false;
            }
            //urls
            if (empty($checkout_url) || !Validate::isAbsoluteUrl($checkout_url)) {
                $output = $this->displayError($this->l('Invalid checkout url'));
                $validator &= false;
            }

            if ($validator) {
                foreach (array_keys($this->configPreset) as $key) {
                    Configuration::updateValue($key, Tools::getValue($key));
                }
                Configuration::updateValue('WPG_METHOD', serialize($payment_method));
                $output = $this->displayConfirmation($this->l('Settings updated'));
            }
        }
        return $output . $this->displayForm();
    }

    public function displayForm()
    {
        $options_method = array(
            array(
                'id_method'     => 'card',
                'name_method'   => 'Card',
            ),
            array(
                'id_method'     => 'googlepay',
                'name_method'   => 'GooglePay',
            ),
            array(
                'id_method'     => 'paypal',
                'name_method'   => 'PayPal',
            ),
        );

        $form = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->l('Title'),
                        'name' => 'WPG_TITLE',
                        'size' => 15,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Merchant key'),
                        'name' => 'WPG_MERCHANT_KEY',
                        'size' => 36,
                        'required' => true,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Merchant password'),
                        'name' => 'WPG_MERCHANT_PASSWORD',
                        'size' => 40,
                        'required' => true,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Checkout host'),
                        'name' => 'WPG_CHECKOUT',
                        'required' => true,
                        'desc' => $this->l('Url from payment system to send a payment request'),
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Callback URL'),
                        'name' => 'WPG_CALLBACK',
                        'required' => false,
                        'desc' => $this->l('Provide this link in the payment system'),
                    ],
                    [
                        'type' => 'select',
                        'label' => $this->l('Payment method'),
                        'name' => 'WPG_METHOD',
                        //'class' => 'multiselect',
                        'multiple' => true,
                        'options' => [
                            'query' => $options_method,
                            'id'    => 'id_method',
                            'name'  => 'name_method',
                        ],
                        'desc' => $this->l('Select payment methods that must be enabled during the customer\'s Checkout session. Customer will see only selected payment methods. If no selection, then all available payment methods will be shown.'),
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                ],
            ],
        ];
        //end form
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->identifier = $this->identifier;
        $helper->name_controller = $this->name;

        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex . '&' . http_build_query(['configure' => $this->name]);

        $helper->submit_action = 'submit' . $this->name;

        // Default language
        $helper->default_form_language = (int) Configuration::get('PS_LANG_DEFAULT');

        // Load current value into the form
        foreach (array_keys($this->configPreset) as $key) {
            $helper->fields_value[$key] = Tools::getValue($key, Configuration::get($key));
        }
        $helper->fields_value['WPG_METHOD[]'] = unserialize(Configuration::get('WPG_METHOD'));
        $helper->fields_value['WPG_CALLBACK'] = $this->context->link->getModuleLink($this->name, 'callback');

        //$this->logger->log(LogLevel::DEBUG, 'Test ' . print_r(unserialize(Configuration::get('WPG_METHOD')), true));
        return $helper->generateForm([$form]);
    }

    public function hookPaymentOptions()
    {
        if (!$this->active) {
            return;
        }

        $payment_options = [
            $this->getExternalPaymentOption(),
        ];

        return $payment_options;
    }
    public function getExternalPaymentOption()
    {
        $externalOption = new PaymentOption();
        $externalOption->setCallToActionText($this->l(Configuration::get('WPG_TITLE')));
        //$externalOption->setLogo(Media::getMediaPath(_PS_MODULE_DIR_ . $this->name . '/logo.png'));
        $externalOption->setAction($this->context->link->getModuleLink($this->name, 'process'));

        return $externalOption;
    }

    public function hookActionOrderSlipAdd($params)
    {
        if (_PS_MODE_DEV_ == true) {
            $this->logger->log(LogLevel::DEBUG, 'RefundData: ' . print_r($params['productList'], true));
        }

        //if another payment method
        if ($params['order']->module != $this->name) {
            if (_PS_MODE_DEV_ == true) {
                $this->logger->log(LogLevel::DEBUG, 'ExternalPaymentOptions: Non this module. Name: ' . $params['order']->module);
            }
            return false;
        }

        //get paymet id
        $order_id = $params['order']->id;
        $order = new Order((int)$order_id);
        $order_payment = OrderPayment::getByOrderReference($order->reference);
        $payment_id = $order_payment[0]->transaction_id;

        if (empty($payment_id)) {
            if (_PS_MODE_DEV_ == true) {
                $this->logger->log(LogLevel::DEBUG, 'Refund: An error has occured. No payment id found');
            }
            $this->addFlash('error', 'An error has occured. No payment id found');
            throw new OrderException('You cannot generate a partial credit slip.');
        }

        $amount = $this->getStandardRefundAmount($params);

        //prepare refund
        $param = array(
            'merchant_key' => Configuration::get('WPG_MERCHANT_KEY'),
            'payment_id'   => $payment_id,
            'amount'       => number_format($amount, 2, '.', ''),
            'hash'         => sha1(md5(Tools::strtoupper($payment_id . number_format($amount, 2, '.', '') . Configuration::get('WPG_MERCHANT_PASSWORD'))))
        );

        $json = json_encode($param);
        if (_PS_MODE_DEV_ == true) {
            $this->logger->log(LogLevel::DEBUG, 'RefundJson: '.$json);
        }
        $link = rtrim(Configuration::get('WPG_CHECKOUT'), '/').$this->apiMethod['refund'];
        $getter = curl_init($link); //init curl
        curl_setopt($getter, CURLOPT_POST, 1); //post
        curl_setopt($getter, CURLOPT_POSTFIELDS, $json); //json
        curl_setopt($getter, CURLOPT_HTTPHEADER, array('Content-Type:application/json')); //header
        curl_setopt($getter, CURLOPT_RETURNTRANSFER, true);

        $result = curl_exec($getter);
        $httpcode = curl_getinfo($getter, CURLINFO_HTTP_CODE);

        if (_PS_MODE_DEV_ == true) {
            //$response = json_decode($result, true);
            $this->logger->log(LogLevel::DEBUG, 'RefundHttpCode: '.$httpcode);
            $this->logger->log(LogLevel::DEBUG, 'RefundJson: '.$result);
        }
        $response = json_decode($result, true);

        if (isset($response['errors'])) {
            foreach ($response['errors'] as $mes) {
                $this->addFlash('error', $mes['error_message']);
            }
            throw new OrderException('You cannot generate a partial credit slip.');
        }
        //actionProductCancel
        //orederdefaultupdater
    }
    private function getStandardRefundAmount($productList)
    {
        //cancel_product[shipping_amount]
        if (_PS_MODE_DEV_ == true) {
            $this->logger->log(LogLevel::DEBUG, 'PostData: ' . print_r($_POST, true));
        }
        $order_shipping_cost_amount = $productList['order']->total_shipping_tax_incl;//order shipping cost

        $shipping_cost_amount = 0;
        $amount_tax_incl = 0;

        //get shipping amount partial refund
        $cancelProduct = Tools::getValue('cancel_product');
        //$shipping_cost_amount = $cancelProduct['shipping_amount'] ? $cancelProduct['shipping_amount'] : false;
        if (isset($cancelProduct['shipping_amount'])) {
            if ($cancelProduct['shipping_amount'] > $order_shipping_cost_amount) {
                $shipping_cost_amount += $order_shipping_cost_amount;
            } else {
                $shipping_cost_amount += $cancelProduct['shipping_amount'];
            }
        }
        //get shipping amount standart refund
        if (isset($cancelProduct['shipping']) && $cancelProduct['shipping'] == 1) {
            //$this->total_shipping_tax_incl
            $shipping_cost_amount = $order_shipping_cost_amount;
        }


        
        foreach ($productList['productList'] as $productRefund) {
            $amount_tax_incl += $productRefund['total_refunded_tax_incl'];
        }

        if (_PS_MODE_DEV_ == true) {
            $this->logger->log(LogLevel::DEBUG, 'RefundAmount: ' .$amount_tax_incl.':'.$shipping_cost_amount);
        }
        return $amount_tax_incl + $shipping_cost_amount;
    }
    protected function addFlash($type, $message)
    {
        $container = $this->getContainer();
        if (!$container->has('session')) {
            throw new \LogicException('You can not use the addFlash method if sessions are disabled. Enable them in "config/packages/framework.yaml".');
        }
        $container->get('session')->getFlashBag()->add($type, $message);
    }
}
